// popup.js (FULL) ✅ with Undo button (undo delete + undo clear all)

async function getActiveTab() {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    return tab;
}

function shortText(t) {
    const s = (t || "").trim().replace(/\s+/g, " ");
    return s.length > 140 ? s.slice(0, 140) + "…" : s;
}

// ---------------- STATUS HELPER (restore previous text) ----------------
let statusTimer = null;

function showStatus(statusEl, text, ms = 1000) {
    if (!statusEl) return;

    const prev = statusEl.textContent;

    if (statusTimer) clearTimeout(statusTimer);

    statusEl.textContent = text;

    statusTimer = setTimeout(() => {
        statusEl.textContent = prev;
        statusTimer = null;
    }, ms);
}
// ----------------------------------------------------------------------

// ---------------- UNDO STACK ----------------
let undoStack = []; // each item: { type: "REMOVE_ONE"|"CLEAR_ALL", records: [...] }

function setUndoEnabled(btn) {
    if (!btn) return;
    btn.disabled = undoStack.length === 0;
}
// -------------------------------------------

// Inject content script if needed, then resend message
async function sendToContent(tabId, message) {
    try {
        return await chrome.tabs.sendMessage(tabId, message);
    } catch (e) {
        await chrome.scripting.executeScript({
            target: { tabId },
            files: ["content.js"],
        });
        return await chrome.tabs.sendMessage(tabId, message);
    }
}

async function copyToClipboard(text) {
    try {
        await navigator.clipboard.writeText(text);
        return true;
    } catch {
        const ta = document.createElement("textarea");
        ta.value = text;
        document.body.appendChild(ta);
        ta.select();
        const ok = document.execCommand("copy");
        document.body.removeChild(ta);
        return ok;
    }
}

function render(listEl, statusEl, badgeEl, items) {
    listEl.innerHTML = "";

    const count = items?.length || 0;
    if (badgeEl) badgeEl.textContent = String(count);

    if (!items || count === 0) {
        statusEl.textContent = "No highlights on this page.";
        const empty = document.createElement("div");
        empty.className = "empty";
        empty.textContent =
            "Select any text on the page to highlight it. Then manage and copy highlights here.";
        listEl.appendChild(empty);
        return;
    }

    statusEl.textContent = `Highlights: ${count}`;

    for (const item of items) {
        const row = document.createElement("div");
        row.className = "item";

        const left = document.createElement("div");

        const text = document.createElement("div");
        text.className = "text";
        text.textContent = shortText(item.text);

        left.appendChild(text);

        const actions = document.createElement("div");
        actions.className = "actions";

        // -------- delete ----------
        const del = document.createElement("button");
        del.className = "btn danger small";
        del.textContent = "unmark";
        del.addEventListener("click", async () => {
            try {
                const tab = await getActiveTab();

                const res = await sendToContent(tab.id, {
                    type: "REMOVE_HIGHLIGHT",
                    id: item.id,
                });

                if (res?.ok) {
                    // ✅ store deleted record for UNDO
                    if (res?.deletedRecord) {
                        undoStack.push({ type: "REMOVE_ONE", records: [res.deletedRecord] });
                    }
                    setUndoEnabled(document.getElementById("undoBtn"));

                    const again = await sendToContent(tab.id, {
                        type: "GET_HIGHLIGHTS",
                    });
                    render(listEl, statusEl, badgeEl, again?.items || []);

                    showStatus(statusEl, "Deleted");
                } else {
                    showStatus(statusEl, "delete failed.");
                }
            } catch {
                showStatus(statusEl, "delete failed.");
            }
        });

        // -------- copy one ----------
        const copyBtnOne = document.createElement("button");
        copyBtnOne.className = "btn small";
        copyBtnOne.textContent = "copy";
        copyBtnOne.addEventListener("click", async () => {
            const toCopy = (item.text || "").trim();
            if (!toCopy) {
                showStatus(statusEl, "Nothing to copy.");
                return;
            }

            const ok = await copyToClipboard(toCopy);
            showStatus(statusEl, ok ? "Copied" : "copy failed.");
        });

        actions.appendChild(del);
        actions.appendChild(copyBtnOne);

        row.appendChild(left);
        row.appendChild(actions);
        listEl.appendChild(row);
    }
}

document.addEventListener("DOMContentLoaded", async () => {
    const listEl = document.getElementById("list");
    const statusEl = document.getElementById("status");
    const badgeEl = document.getElementById("badge");

    const clearBtn = document.getElementById("clearAll");
    const copyBtn = document.getElementById("copyAll");
    const undoBtn = document.getElementById("undoBtn");

    setUndoEnabled(undoBtn);

    // Initial load
    try {
        const tab = await getActiveTab();
        const res = await sendToContent(tab.id, { type: "GET_HIGHLIGHTS" });
        render(listEl, statusEl, badgeEl, res?.items || []);
    } catch {
        statusEl.textContent = "Cannot access this page (try a normal webpage tab).";
        if (badgeEl) badgeEl.textContent = "0";
    }

    // -------- UNDO ----------
    undoBtn?.addEventListener("click", async () => {
        const action = undoStack.pop();
        setUndoEnabled(undoBtn);

        if (!action) {
            showStatus(statusEl, "Nothing to undo.");
            return;
        }

        try {
            const tab = await getActiveTab();

            for (const rec of action.records) {
                await sendToContent(tab.id, { type: "RESTORE_HIGHLIGHT", record: rec });
            }

            const again = await sendToContent(tab.id, { type: "GET_HIGHLIGHTS" });
            render(listEl, statusEl, badgeEl, again?.items || []);

            showStatus(statusEl, "Undo complete");
        } catch {
            showStatus(statusEl, "Undo failed.");
        }
    });

    // -------- clear all ----------
    clearBtn?.addEventListener("click", async () => {
        try {
            const tab = await getActiveTab();

            // ✅ save all records for UNDO
            const before = await sendToContent(tab.id, { type: "GET_RECORDS" });
            const records = before?.records || [];
            if (records.length) {
                undoStack.push({ type: "CLEAR_ALL", records });
                setUndoEnabled(undoBtn);
            }

            await sendToContent(tab.id, { type: "CLEAR_ALL" });

            const res = await sendToContent(tab.id, { type: "GET_HIGHLIGHTS" });
            render(listEl, statusEl, badgeEl, res?.items || []);
            showStatus(statusEl, "All highlights cleared.");
        } catch {
            showStatus(statusEl, "Failed to clear highlights.");
        }
    });

    // -------- copy all ----------
    copyBtn?.addEventListener("click", async () => {
        try {
            const tab = await getActiveTab();
            const res = await sendToContent(tab.id, { type: "COPY_ALL" });
            const text = res?.text || "";

            if (!text.trim()) {
                showStatus(statusEl, "Nothing to copy.");
                return;
            }

            const ok = await copyToClipboard(text);
            showStatus(statusEl, ok ? "Copied all highlights!" : "copy failed.");
        } catch {
            showStatus(statusEl, "copy failed.");
        }
    });
});
